#!/bin/ash
#
#
#==========================================================================
#
#   = FILENAME  crm/crm_upgrade/package/firmware/init.sh
#     init - script for upgrading fan firmware, Ahi FPGA, and u-boot/rfs/kernel
#
#   = COPYRIGHT
#     Copyright Agilent Technologies 2011
# Invoked from:
#    /opt/agilent/bin/initpkghelper ftp://169.254.226.149/crm032/signed_script_package.tar.gz
#
#========================================================================== 
${DEBUG:=false} && set -x 
ROOTDIR=""   # used for testing by on a relative root directory structure

CMD_NAME=$(basename $0 | tr [:upper:] [:lower:])
CURRENT_DIR=$(pwd)
PACKAGE_DIR=$(dirname $0)

. /etc/profile

cd $PACKAGE_DIR


func_usage()
{
 : ENTERING: func_usage $*
 exec 1>&2
 [ -n "$*" ] && echo "$CMD_NAME: $*"
 echo "Usage: $CMD_NAME [-D] [-t <rootdir>] [-e] [-v] <ftphostdir>" 
 echo ""
 echo "  -D --  Enable set -x script debugging"
 echo "  -t <rootdir>  Test using a different root directory."
 echo "  -e --  Examination install.  Don't actually install anything; just see actions to be performed."
 echo "         Ignores the -f option"
 echo "  -f --  Force install.  Update everything."
 echo "  -v --  Be more verbose."
 echo " <ftphostdir> -- The ftp://<hostname>/<dirpath> where the firmware bits are pulled from via ftp."
 echo "                 For example: ftp://169.254.226.149/crm032"
 exit 1
}


#======================================================
# Description: 
# tee useful gappLogger messages to stdout too!
#======================================================
func_logit ()
{
  : ENTERING: func_logit $*
  if [ "$1" = "-p" ] ; then
    echo "logger: $0 $*";
    $gappLogger $1 $2 "$3";
  else  
    echo "logger: $0: $*";
    $gappLogger "$*";
  fi;
  [ "$examineInstall" = "false" ] && echo "$*" >> $STATUS_NONVOL_FILE;
  return 0
}

#======================================================
# Description: 
# Determine the class of chassis
#======================================================
func_getChassisClass()
{
  : ENTERING: func_getChassisClass 
  if uname -r | grep AXIE > /dev/null
  then
    gchassisClass=AXIE
  else 
    gchassisClass=AMP
  fi
}

#======================================================
# Description: 
# Determine the type of chassis
#======================================================
func_getChassisType()
{
  : ENTERING: func_getChassisType
  gchassisType=`cat $chassisTypeFile`;
  if [ "$gchassisType" != "F2A" ] &&  [ "$gchassisType" != "F14A" ]  && \
     [ "$gchassisType" != "F2AX" ] && [ "$gchassisType" != "F14AX" ] && \
     [ "$gchassisType" != "F5AX" ]
  then
    func_logit "INSTALL: Warning: Unrecognized chassis type: $gchassisType";
    exit 1
  fi
}


#======================================================
# Description: 
# Function to check the md5dsum and filesize
#======================================================
func_binaryCheck ()
{
    : ENTERING: func_binaryCheck $1
    binaryfile="$1"
    [ "$binaryfile" = "none" ] && return

    pkgChksum=`cat ./filemeta.txt | grep "$binaryfile" | awk '{print $1}'`

    fileChksum=`wget -qO - "$ftphostdir/$binaryfile" | md5sum | awk '{print $1}'`;
    if [ "$fileChksum" != "$pkgChksum" ] ; then
       func_logit -p user.err "INSTALL:   Checksum mismatch detected for $fileChksum"
       exit 1
    fi

    func_logit "   $binaryfile package verified."; 
}


#======================================================
# Description: 
# Check available disk space for listed files $0
#======================================================
func_checkFreeSpace()
{
  : ENTERING: func_checkFreeSpace 
  mfreeblocks=`df -m | grep "/dev/ram0" | awk '{print $4}'`;
  freespacebytes=`expr $mfreeblocks \* 1024 \* 1024`;
  bytesRequired=0

  for i in $*
  do
    if [ "$i" != "none" ]
    then
      isize=`wget -qO - "$ftphostdir/$i" | wc -c`;
      bytesRequired=`expr $bytesRequired + $isize`;
    fi
  done

  if [ $bytesRequired -gt $freespacebytes ] ; then
    func_logit -p user.err "INSTALL: Not enough ramdisk space for upgrade.  Required: $totalimagesize, Free: $freespacebytes";
    exit 1;
  fi
}


#======================================================
# Description: 
# Download listed file $1 (from) $2 (to)
#======================================================
func_downloadFile()
{
  : ENTERING: func_downloadFile $1 $2
  srcFile=$1
  tgtFile=$2

  [ "$srcFile" = "none" ] && return

  # Get file over FTP
  func_logit "  downloading $srcFile...";
  wget -qO  "$tgtFile" "$ftphostdir/$srcFile" 1>/dev/null 2>&1 ;

  # Check that the files retrieved are actually present
  if [ ! -f "$tgtFile" ]
  then
    func_logit -p user.err "INSTALL: ERROR: $tgtFile does not exist";
    exit 1;
  fi
}


#======================================================
# Description: 
# Get list of blades with a particular attribute.
#======================================================
getAmpBladesWithAttribute ()
{    
    : ENTERING: getAmpBladesWithAttribute
    # Get list of active blades
    bladelist=`clia board | grep "Physical Slot" | cut -f 4 -d " "`
    finalBladeList=""

    # Loop through each of the blades, and examine their OEM fru
    for blade in $bladelist; 
    do 
       boardAddr=`clia board $blade | grep "FRU # 0" | awk -F : '{print $1}'`
       bladeAttr=`clia fruinfo -v -x $boardAddr 0 | awk -f ${ROOTDIR}/opt/agilent/bin/amp-oemparse.awk | grep "AMP_BLADE_ATTRIBUTE" | awk -F = '{print $2}'`       

       # Just check for the existence of the AMP_BLADE_ATTRIBUTE oem record
       if [ ! -z "${bladeAttr}" ] 
       then
            finalBladeList="${finalBladeList}${blade} "
       fi
    done

    echo "${finalBladeList}"
}

#======================================================
# Description: 
# Get list of blades that need to be shut down during
# upgrade
#======================================================
getAmpBlades ()
{    
    : ENTERING: getAmpBlades
    bladelist=`getAmpBladesWithAttribute`
    echo "${bladelist}"
}

#======================================================
# Description: 
# Cleanup routine and then reboot
#======================================================
func_cleanupRebootTrap ()
{
   : ENTERING: func_cleanupRebootTrap 
   tempdir="$1"	
	rm -rf "$tempdir";
	cat "${ROOTDIR}/var/log/messages" | grep "shm-upgrade\|ahi-fpga-program" > ${upgradeStatusFile}
   func_logit "INSTALL: Rebooting firmware....";
   func_logit "CGI:end_update";
   func_logit "";
   sleep 2;
   
   [ "$gbool_complete_rupgrade" = "true" ] && rupgrade_tool -s -R;

#   echo "DEN-Reboot is commented out"
	reboot
}

#======================================================
# Description: 
# Cleanup routine and then reboot
#======================================================
func_cleanupExitTrap ()
{
   : ENTERING: func_cleanupExitTrap
   echo "Cleaning up...."
	tempdir="$1"	
	rm -rf "$tempdir";

	cat "${ROOTDIR}/var/log/messages" | grep "shm-upgrade\|ahi-fpga-program" > ${upgradeStatusFile}

   if [ "$gbool_complete_rupgrade" = "true" ]
   then
     func_logit "INSTALL: Rebooting firmware...";
     func_logit "CGI:end_update";
     func_logit "";
     sleep 2;

     rupgrade_tool -s -R;

   else

     [ "$(func_shelfManagerAlive)" = "false" ] && func_startShelfManager
     [ "$(func_ATBAlive)" = "false" ] && func_startATB
   
     func_logit "INSTALL: Exiting...";
     
     func_logit "CGI:end_update";
     func_logit "";
     sleep 2;
   fi
}



##############################################################################
# Pre-requisite setup prior to the first actual install of something--only need to do once.
##############################################################################
func_preInstallSetup()
{

  : ENTERING: func_preInstallSetup 
  
  if [ "$gbool_EnvSetupDone" = "false" ]
  then

    func_logit "INSTALL: Preparing Environment";

    # Override and reset default u-boot variables
    func_logit "  updating environment. (ipdevice)";
    setenv ipdevice lo:1
    func_logit "  updating environment.. (rc2_daemons)";
    setenv start_rc2_daemons y
    #setenv rc2 ""
    func_logit "  updating environment...(rc2)";
    setenv rc2 "${ROOTDIR}/etc/rc.axie"
    func_logit "  updating environment....(quiet)";
    setenv quiet quiet
    gbool_EnvSetupDone=true
  fi

}


##############################################################################
# USB firmware has no revision read out feature like ATB.  This check
# allows a way to skip over the 40 minute update or erase the contents of the 
# USB functionality if there is interference with PCIe connectivity.
##############################################################################
func_check_usb()
{
  wget -qO "$temp/progusb" "$ftphostdir/progusb" 1>/dev/null 2>&1 ;
  gbool_skip_usb=true;
  gbool_skip_usb_but_erase=false;

  if [ -f "$temp/progusb" ]
  then
    gbool_skip_usb=false;
    if [ "$(grep -c "erase" "$temp/progusb")" = "1" ]
    then
      gbool_skip_usb_but_erase=true
    fi;
  fi;
}


##############################################################################
# Pre-requisite checks before we try the upgrade
##############################################################################
func_preUpgradeCheck()
{

  # Check the existence of dependency files on RFS before proceeding with upgrade
  if [ ! -f "${chassisTypeFile}" ]
  then
    func_logit "INSTALL: ERROR: ${chassisTypeFile} does not exist";
    exit 1;
  fi

  if [ ! -f "${atbPeekPokeUtil}" ]
  then
    func_logit "INSTALL: Warning: ${atbPeekPokeUtil} does not exist";
#    exit 1;
  fi

  if [ ! -f "${fpgaUpgradeUtil}" ]
  then
    func_logit "INSTALL: Warning: ${fpgaUpgradeUtil} does not exist";
#    exit 1;
  fi
  
  func_logit "INSTALL:  Determining chassis class and type...";
  func_getChassisClass
  func_getChassisType
  func_logit "INSTALL:  Chassis Details:"
  varfile="/tmp/chassis_var"
  func_logit $(grep z_chassis_model    $varfile | awk -F'\"' '{print "INSTALL:    Product: ", $2}')
  func_logit $(grep z_chassis_sn       $varfile | awk -F'\"' '{print "INSTALL:    SN: ", $2}')
  func_logit $(grep z_chassis_revision $varfile | awk -F'\"' '{print "INSTALL:    Revision before update: ", $2}')

}

################################################################################
#
# Function: version_check
# Description: Version check of the firmware package
#
################################################################################
version_check()
{
  : ENTERING: version_check
  # Check the version to see if we need to do an upgrade
  # The CRM firmware version can be obtained from the uname, while the 
  # fan version information must be retrieved dynamically

  gchassisType=`cat $chassisTypeFile`;
  if [ "$gchassisType" != "F2A" ] &&  [ "$gchassisType" != "F14A" ]  && \
     [ "$gchassisType" != "F2AX" ] && [ "$gchassisType" != "F14AX" ] && [ "$gchassisType" != "F5AX" ]
  then
    func_logit "INSTALL: Warning: Unrecognized chassis type: $gchassisType";
    exit 1
  fi

  if uname -r | grep AXIE > /dev/null
  then
    gchassisClass=AXIE
  else 
    gchassisClass=AMP
  fi

# DEN Look for non-volatile revision file if there use that else use the following:
  crmFirm=`uname -r    | awk -F ${gchassisClass}. '{print $2}' | awk -F - '{print $1}'`
  fan2aFirm=`uname -r  | awk -F ${gchassisClass}. '{print $2}' | awk -F - '{print $2}'`
  fan14aFirm=`uname -r | awk -F ${gchassisClass}. '{print $2}' | awk -F - '{print $3}'`
  fan5aFirm=`uname -r  | awk -F ${gchassisClass}. '{print $2}' | awk -F - '{print $4}'`



  firstFan=`clia fans  | grep FRU | cut -f1 -d':' | head -n 1`
  fanCurrentResp=`clia sendcmd $firstFan 6 1 | grep "Response"`
  fanCurrentStatus=$?
  fanCurrentFirm=""

  # Get the base fan firmware used for comparison - if we can't get it, use the one found in the uname
  if [ "$fanCurrentStatus" != "0" ]
  then  # use static uname version
    case "$gchassisType" in
      F2A|F2AX)
        fanCurrentFirm="$fan2aFirm"
        ;;

      F5AX)
        fanCurrentFirm="$fan5aFirm"
        ;;

      F14A|F14AX)
        fanCurrentFirm="$fan14aFirm"
        ;;

      *)
        func_logit "INSTALL: Warning: Failed to match chassis type"
        exit 1
        ;;

    esac


    # Strip off leading F2A or F14A, etc
    fanCurrentFirm=`echo "$fanCurrentFirm" | awk -F . '{print $2}'`
    func_logit -p user.err "INSTALL: Warning: Cannot get fan information for fan: $firstFan";
  else
    fanCurrentFirm=`echo "$fanCurrentResp" | awk '{print $5$6}'`
  fi
    
  echo "Chassis current revision is: $fanCurrentFirm"

  # Query the fan version dynamically, and check for mismatchs if we have
  # multiple fans
  for fan in `clia fans | grep FRU | cut -f1 -d':'`; 
  do 
    fanState=`clia fru $fan | grep "Hot Swap State" | awk '{print $4}'`;
    if [ "$fanState" != "M4" ] ; then
       continue;
    fi

    faniresp=`clia sendcmd $fan 6 1 | grep "Response"`;
    fanistatus=$?
    if [ "$fanistatus" != "0" ]
    then
      func_logit -p user.err "INSTALL: Warning: Cannot et fan information for fan: $fan";
      continue;
    fi

    faniFirm=`echo "$faniresp" | awk '{print $5$6}'`;

    # If we detect any of the fans that don't match, set it back to empty string
    # and log a message to syslog
    if [ "$fanCurrentFirm" != "$faniFirm" ] ; then
      func_logit "Fans detected to have different firmware versions: $fanCurrentFirm and $faniFirm";
      fanCurrentFirm="";
      break;
    fi
  done


  case "$gchassisType" in
    F2A|F2AX)
   fan2aFirm="$gchassisType.$fanCurrentFirm";
      ;;

    F5AX)
      fan5aFirm="$gchassisType.$fanCurrentFirm";
      ;;

    F14A|F14AX)
      fan14aFirm="$gchassisType.$fanCurrentFirm";
      ;;

    *)
      func_logit "INSTALL: Warning: Failed to match chassis type (2)"
      exit 1
      ;;

  esac
     
     
  # Use the information to generate current ShM firmware version.
  currentVer="$gchassisClass.$crmFirm-$fan2aFirm-$fan14aFirm"    # DEN Need another fan version identifier

  if test -z "$currentVer"; then
    version=0;
    func_logit 'INSTALL: Warning: Unable to determine installed ShMM firmware version';
  fi;

  # Get the version of the upgrade package
  packageVer=$(cat $pkgVersionFile | grep '^Version:' | awk '{print $2}')

  
  echo "Package Revision:   $packageVer"
  echo "Installed Revision: $currentVer"

  
  # DEN Need to differentiate AMP vs AXIE package...initpkhelper could ask for appropriate package
  if test "$currentVer" != "$packageVer"; then
    func_logit "Upgrading ShMM firmware to $packageVer (current version: $currentVer)";
  else
    # No upgrade needed.
    func_logit "ShMM firmware is up to date.";
    exit 0;
  fi;

  # Create a temp dir where we will be downloading all our files
  temp="$PWD/tmp";
  if ! mkdir -m 700 "${temp}" ; then
    func_logit -p user.err "INSTALL: ERROR: Failed to create dir: ${temp}";
    exit 1;
  fi;

  # At this point, we know that we will proceed with the upgrade.
  # Setup the cleanup trap, and remove the normalBoot cookie file
  # in nvram
  rm -rf "${normalBootCookie}"

  # Trap for cleaning up
#  trap 'func_cleanupRebootTrap "$temp"' INT TERM QUIT EXIT

}

################################################################################
#
# Function: func_lookupInstallFile
# Description: return the filename for the token name descriptor $1
#
################################################################################
func_lookupInstallFile()
{
  : ENTERING: func_lookupInstallFile $1
  MANIFEST="packages"
  [ "$1" = "" ] && func_logit "INSTALL: ERROR: func_lookupInstallFile: filename not specified" && exit 1

  pkgFile=$(grep "${1}::" $MANIFEST | awk '{print $2}')
  if [ -z "$pkgFile" ]
  then
    func_logit "INSTALL: ERROR: package not found in manifest file \"$MANIFEST\" for $1" 
    exit 1
    pkgFile="pkg_name_not_found"
  fi

  echo "$pkgFile"
  unset pkgFile
}



################################################################################
#
# Function: func_getFanFilename
# Description: Return the filename for the fan package based on requested 
#              chassisType $1: gchassisType may be F2A, F5AX, etc
#
################################################################################
func_getFanFilename()
{
  : ENTERING: func_getFanFilename 

  [ "$1" = "" ] && func_logit "INSTALL: ERROR: func_getFanFilename: filename not specified" && exit 1
  fanImageName=$(func_lookupInstallFile $1)

  if [ "$fanImage" = "pkg_name_not_found" ]
  then
    func_logit "INSTALL: ERROR: Unknown chassis type or fan package not found: $gchassisType";
    exit 1;
  else
    echo "$fanImageName"
  fi;
  unset fanImageName
}


################################################################################
#
# Function: func_getXSVF_Filenames
# Description: Identify the JTag files for erase and program on ESM
#
################################################################################
func_getXSVF_Filenames()
{
   : ENTERING: func_getXSVF_Filenames
   # DEN there is not a method to differentiate amp crm to axie esm
   # This scan requires ATB services shutdown!!!!   
    # DEN how many devices in chain to determine number of downloads, etc...
    chainLength=$($fpgaUpgradeUtil -verifychain | grep "devices in chain:" | awk '{print $8}')
    if [ "$chainLength" = "4" ]
    then
      # Ahi image names for 4 chain setup
      ahi_erase_tar="$(func_lookupInstallFile  ahi_erase_4)"
      ahi_prog_tar="$(func_lookupInstallFile   ahi_prog_4)"
      usb_erase_tar="$(func_lookupInstallFile  usb_erase_4)"
      usb_prog_tar="$(func_lookupInstallFile   usb_prog_4)"

    elif  [ "$chainLength" = "2" ]
    then
      # Ahi image names for 2 chain setup 
      ahi_erase_tar="$(func_lookupInstallFile ahi_erase_2)"
      ahi_prog_tar="$(func_lookupInstallFile  ahi_prog_2)"

    else
      func_logit -p user.err "INSTALL: $lLabel chain length error of size: $chainLength";
      func_logit -p user.err "INSTALL:   Verify switch settings on ESM silkscreen for ShMM Jtag port";
      exit 1
    fi
}



###############################################################################
# 
###############################################################################
################################################################################
#
# Function: filesize_check
# Description: Check file sizes and checksums before committing them to ramdisk
#
################################################################################
filesize_check()
{
  : ENTERING: filesize_check
  # Image names for uboot, rfs, and kernel
  kernel=$(func_lookupInstallFile kernel);
  rootfs=$(func_lookupInstallFile rootfs);
  ubootb=$(func_lookupInstallFile u-boot);

  # Ahi image names
  func_getXSVF_Filenames

  # Image names for fan
  fanImage="$(func_getFanFilename $gchassisType)"

  mfreeblocks=`df -m | grep "/dev/ram0" | awk '{print $4}'`;
  freespacebytes=`expr $mfreeblocks \* 1024 \* 1024`;

  ipmisize=`wget -qO - "$ftphostdir/ipmitool" | wc -c`;
  fansize=`wget -qO - "$ftphostdir/$fanImage" | wc -c`;
  erasexsvfsize=`wget -qO - "$ftphostdir/$ahi_erase_tar" | wc -c`;
  ahixsvfsize=`wget -qO -   "$ftphostdir/$ahi_prog_tar"  | wc -c`;
  totalimagesize=`expr $ipmisize + $fansize + $erasexsvfsize + $ahixsvfsize`;

  if [ $totalimagesize -gt $freespacebytes ] ; then
    func_logit -p user.err "INSTALL: ERROR: Not enough ramdisk space for upgrade.  Required: $totalimagesize, Free: $freespacebytes";
    exit 1;
  fi

  # Download files, and verify checksum
  ipmisum=`wget -qO - "$ftphostdir/ipmitool" | md5sum | awk '{print $1}'`;
  func_binaryCheck "ipmitool" "$ipmisum";

  fansum=`wget -qO - "$ftphostdir/$fanImage" | md5sum | awk '{print $1}'`;
  func_binaryCheck "$fanImage" "$fansum";

  erasexsvfsum=`wget -qO - "$ftphostdir/$ahi_erase_tar" | md5sum | awk '{print $1}'`;
  func_binaryCheck "$ahi_erase_tar" "$erasexsvfsum";

  ahixsvfsum=`wget -qO - "$ftphostdir/$ahi_prog_tar" | md5sum | awk '{print $1}'`;
  func_binaryCheck "$ahi_prog_tar" "$ahixsvfsum";

  ubootsum=`wget -qO - "$ftphostdir/$ubootb" | md5sum | awk '{print $1}'`;
  func_binaryCheck "$ubootb" "$ubootsum";

  rootfssum=`wget -qO - "$ftphostdir/$rootfs" | md5sum | awk '{print $1}'`;
  func_binaryCheck "$rootfs" "$rootfssum";

  kernelsum=`wget -qO - "$ftphostdir/$kernel" | md5sum | awk '{print $1}'`;
  func_binaryCheck "$kernel" "$kernelsum";

  func_logit "Upgrade image size and checksum verification complete.";
}


################################################################################
#
# Function: version_check
# Description: Version check of the firmware package
#   should return a fanFirm and fanPackage settings
#
# DEN: May want to include using clia fruinfo c0 0
#      which lists product and board info areas. The Product Version field lists
#      useful identifers such as AX5-FAN-ES1.E0M0 so we don't accidentally update
#      an amp fan with axie fan bits.
#
################################################################################
func_fanVersionCheck()
{
  : ENTERING: func_fanVersionCheck
  # Check the version to see if we need to do an upgrade
  # The CRM firmware version can be obtained from the uname, while the 
  # fan version information must be retrieved dynamically

  func_getChassisType
  func_getChassisClass

  if [ "$(func_shelfManagerAlive)" = "false" ]
  then
    func_startShelfManager
  fi
 

  # Look up in old place for an AMP chassis
  if [ "$gChassisClass" = "AMP" ]
  then
    # Get the base fan firmware used for comparison - if we can't get it, use the one found in the uname
    case "$gchassisType" in
      F2A)
        fanFirmDefault=`uname -r  | awk -F ${gchassisClass}. '{print $2}' | awk -F - '{print $2}'`
        ;;

      F14A)
        fanFirmDefault=`uname -r | awk -F ${gchassisClass}. '{print $2}' | awk -F - '{print $3}'`
        ;;

      *)
        func_logit "INSTALL: Warning: Failed to match AMP chassis type: $gchassisType"
        exit 1
        ;;
    esac
    # Strip off leading F2A or F14A, etc
    fanFirmDefault=`echo "$fanFirmDefault" | awk -F . '{print $2}'`

  else
    fanFirmDefault=$( cat $installedVersionFile | grep "^${gchassisType}:" | awk '{print $2}')
    fanPackage=$(     cat       $pkgVersionFile | grep "^${gchassisType}:" | awk '{print $2}')
  fi



  firstFan=`clia fans  | grep FRU | cut -f1 -d':' | head -n 1`
  fanCurrentResp=`clia sendcmd $firstFan 6 1 | grep "Response"`
  fanCurrentStatus=$?
  fanCurrentFirm=""

  # Get the base fan firmware used for comparison - if we can't get it, use the OS referenced version
  if [ "$fanCurrentStatus" != "0" ]
  then  # use static uname version
    fanFirm=$fanFirmDefault
    func_logit -p user.err "INSTALL: Warning(1): cannot get fan information for fan: $firstFan";
  else
    fanFirm=`echo "$fanCurrentResp" | awk '{print $5$6}'`
  fi
    
  #echo "Chassis current revision is: $fanFirm"

  # Query the fan version dynamically, and check for mismatchs if we have
  # multiple fans
  for fan in `clia fans | grep FRU | cut -f1 -d':'`; 
  do 
    fanState=`clia fru $fan | grep "Hot Swap State" | awk '{print $4}'`;
    if [ "$fanState" != "M4" ] ; then
       continue;
    fi

    faniresp=`clia sendcmd $fan 6 1 | grep "Response"`;
    fanistatus=$?
    if [ "$fanistatus" != "0" ]
    then
      func_logit -p user.err "INSTALL: Warning(2): cannot get fan information for fan: $fan";
      continue;
    fi

    faniFirm=`echo "$faniresp" | awk '{print $5$6}'`;

    # If we detect any of the fans that don't match, set it back to empty string
    # and log a message to syslog
    if [ "$fanFirm" != "$faniFirm" ] ; then
      func_logit "INSTALL: Warning: Fans detected to have different firmware versions: $fanCurrentFirm and $faniFirm";
      fanCurrentFirm="";
      break;
    fi
  done


 fanImage="$(func_getFanFilename $gchassisType)"

 func_logit "ChassisType: $gchassisType"
 func_logit "Firmware installed: $fanFirm"
 func_logit "Firmware default version: $fanFirmDefault"
 func_logit "Firmware package version: $fanPackage"
 func_logit "FanImage Name:  $fanImage"

# Use the information to generate current ShM firmware version.
#  currentVer="$gchassisClass.$crmFirm-$fan2aFirm-$fan14aFirm"    # DEN Need another fan version identifier

#  if test -z "$currentVer"; then
#    version=0;
#    func_logit 'Warning: Unable to determine installed ShM firmware version';
#  fi;

#  # Get the version of the upgrade package
#  packageVer=$(cat $pkgVersionFile | grep '^Version:' | awk '{print $2}')

  
  # DEN Need to differentiate AMP vs AXIE package...initpkhelper could ask for appropriate package
#  if test "$currentVer" != "$packageVer"; then
#    func_logit "Upgrading ShM firmware to $packageVer (current version: $currentVer)";
#  else
#    # No upgrade needed.
#    func_logit "ShM firmware is up to date.";
#    exit 0;
#  fi;

  # At this point, we know that we will proceed with the upgrade.
  # Setup the cleanup trap, and remove the normalBoot cookie file
  # in nvram
#  rm -rf "${normalBootCookie}"

  # Trap for cleaning up
#  trap 'func_cleanupRebootTrap "$temp"' INT TERM QUIT EXIT

}





################################################################################
#
# Function: func_UpgradeChassis
# Description: Fan firmware upgrade
#
################################################################################
func_Upgrade_Chassis()
{
  

  : ENTERING: func_Upgrade_Chassis $1 $2
  lAction=$1
  lOption=$2
  lLabel="Chassis"

  func_logit "INSTALL: $lLabel: Action: $lAction $lOption";
  if [ "$lAction" = "check_version" ]
  then

    func_fanVersionCheck
    
    if [ "$lOption" = "force" ] || [ "$lOption" = "test" ]
    then
      fanPackage="${fanPackage}_${lOption}"
    fi

    if [ "$fanFirm" != "$fanPackage" ]
    then
      gbool_Install_chassis=true
      func_logit "INSTALL: $lLabel: update required $lOption: (target $fanFirm) != (package $fanPackage)";
    else
      func_logit "INSTALL: $lLabel: version up-to-date: $fanFirm";
    fi

    return
  fi

  if [ "$lAction" = "verify_package" ] || [ "$lAction" = "install" ]  
  then

    func_binaryCheck "ipmitool"
    func_binaryCheck "$fanImage"
    
    
    # Get files over FTP
    func_downloadFile "ipmitool"  "$temp/ipmitool"
    func_downloadFile "$fanImage"  "$temp/$fanImage"
    chmod u+x "$temp/ipmitool";

    func_logit "INSTALL: $lLabel:   package verified and ready to install.";

  
  fi
  
  ipmi_log="${ROOTDIR}/tmp/ipmiupdate.log$$";

  if [ "$lAction" = "install" ]
  then
    if $gbool_Install_chassis || [ "$lOption" = "force" ]
    then
      func_logit "INSTALL: $lLabel: $lOption: Upgrading chassis firmware";
      func_logit "Running $lLabel upgrade..."
      # Perform the upgrade

      fanUpgradeStatus=0    
      shmipaddress=`clia getlanconfig 1 ip | tail -1 | awk '{print $3}'`

      for fan in `clia fans | grep FRU | cut -f1 -d':'`; 
      do 

        fanState=`clia fru $fan | grep "Hot Swap State" | awk '{print $4}'`;
        if [ "$fanState" != "M4" ] && [ "$fanState" != "M3" ] ; then
           func_logit -p user.err "INSTALL: ERROR: Fan $fan expected to be in M3 or M4 state, actual is $fanState";          
           func_logit -p user.err "INSTALL: ERROR: Is fan tray loaded??? Try power cycling chassis.";          
           continue;
        fi

        echo "FAN: $fan" >> $ipmi_log
        if [ "$lOption" = "test" ]
        then
          func_logit "  Test Only..."
          func_logit "    $temp/ipmitool -I lan -H ${shmipaddress} -A NONE -t 0x$fan -b 0 hpm upgrade $temp/$fanImage activate >> $ipmi_log "
        else
#          func_logit "INSTALL: DEN    $temp/ipmitool -I lan -H ${shmipaddress} -A NONE -t 0x$fan -b 0 hpm upgrade $temp/$fanImage activate "
           func_logit "INSTALL: $lLabel: Install step will take approximately 5 minutes...";
           echo "y" | "$temp/ipmitool" -I lan -H ${shmipaddress} -A NONE -t 0x"$fan" -b 0 hpm upgrade "$temp/$fanImage" activate 2>> $ipmi_log 
#DEN Returns code 141. reviewing with PPS           fanUpgradeStatus=$?

        fi

        if test "$fanUpgradeStatus" != 0; then
          func_logit -p user.err "INSTALL: ERROR: ipmitool failed executing fan upgrade for fan:$fan, error code $fanUpgradeStatus";
          exit 1;
        fi;

      done # doing each fan

      # Verification of the fan upgrade:
      func_logit "INSTALL: $lLabel: $lOption: Verify chassis firmware";

#AMP      fanpackage=`cat $pkgVersionFile | grep "Version:" | awk -F - '{print $2}' | awk -F "$gchassisType." '{print $2}'`
#
#      if test $gchassisType = "F14A"; then
#        fanpackage=`cat $pkgVersionFile | grep "Version:" | awk -F - '{print $3}' | awk -F "$gchassisType." '{print $2}'`
#      fi

      fanPackage=$(     cat       $pkgVersionFile | grep "^${gchassisType}:" | awk '{print $2}')
      
      sleep 5
      for fan in `clia fans | grep FRU | cut -f1 -d':'`; 
      do 

        fanState=`clia fru $fan | grep "Hot Swap State" | awk '{print $4}'`;
        if [ "$fanState" != "M4" ] && [ "$fanState" != "M3" ] ; then
           func_logit -p user.err "INSTALL: ERROR: Failed to verify fan $fan. Expected to be in M3 or M4 state, actual is $fanState";          
          continue;
        fi

        # Get the current ShM fan version, example:
        #   pattern: Response data: 12 80 01 08 51 29 00 00 00 02 52 A1 00 00 01
        #                                 Mj Mn
        # Mj-major
        # Mn-minor
        #
        fanCurrent=`clia sendcmd $fan 6 1 | grep "Response data:" | awk '{print $5$6}'`  

        func_logit -p user.err "INSTALL: $lLabel: $lOption: Adjusting fan speed for fan: $fan";

        fanlevel=$(clia fans | grep "Minimum Speed" | sed -e 's/,//' | awk '{print $4}')
        [ -n "$fanlevel" ] && /etc/pfan -amb $fanlevel

        if [ "$lOption" = "test" ]
        then
          func_logit "  Test: verify fanpackage: $fanPackage, installed: $fanCurrent"
        else
          if [ "$fanPackage" != "$fanCurrent" ]
          then
            func_logit -p user.err "INSTALL: ERROR: failed updating fan firmware for fan: $fan. Current version: $fanCurrent (expected version: $fanPackage)";
            echo "Installed information:" >> $ipmi_log
            clia ipmc -v $fan | egrep "Manufacturer ID|Device ID" >> $ipmi_log 
            cat $ipmi_log | sed -e 's/^/INSTALL: Chassis: /'
            rm -f $ipmi_log
            exit 1;
          else
            func_logit -p user.err "INSTALL: $lLabel: install successful to $gchassisType.$fanPackage";
            gbool_reboot=true

          fi
        fi
      
      done # verifying

    else
        func_logit "INSTALL: $lLabel: install not necessary.";
    fi # endif gbool_Install_chassis
  
    # Do some early cleanup to free up space
    rm -f "$temp/$fanImage";
    rm -f "$temp/ipmitool";

  fi # endif lAction = install

}


func_checkChassisBackup()
{
  # Get files over FTP
  func_downloadFile "ipmitool"  "$temp/ipmitool"
  func_downloadFile "$fanImage"  "$temp/$fanImage"
  chmod u+x "$temp/ipmitool";

  shmipaddress=$(clia getlanconfig 1 ip | tail -1 | awk '{print $3}')

  backMajor=$(2>&1 "$temp/ipmitool" -I lan -H ${shmipaddress} -A NONE -t 0xc0 -b 0 hpm compprop 0 3 | awk '/Major:/ {print 0$2}')
  backMinor=$(2>&1 "$temp/ipmitool" -I lan -H ${shmipaddress} -A NONE -t 0xc0 -b 0 hpm compprop 0 3 | awk '/Minor:/ {print 0$2}')
  backRev=${backMajor}${backMinor}
  fanpkg=$(egrep "^${gchassisType}:" $pkgVersionFile | awk '{print $2}')
  if [ "$backRev" != "$fanpkg" ]
  then
    gbool_Install_chassisBackup=true
    func_logit -p user.err "INSTALL: Chassis: provisional update needed from $gchassisType.$backRev to $chassisType:$fanpkg";
  fi

}


func_startATB()
{
  : ENTERING: func_startATB
  # Kill off all instances of atb-protocol agent before proceeding with ahi upgrade
  func_logit "Starting ATB agent and monitor..."
  /etc/agilent_app_monitor &
}


func_ATBAlive()
{
    : ENTERING: func_ATBAlive
    cnt=$(ps | grep agilent_app_monitor | grep -vc grep)
    if [ "$?" = "0" ]
    then
      echo "true"
    else
      echo "false"
    fi
}


func_stopATB()
{
  : ENTERING: func_stopATB
  # Kill off all instances of atb-protocol agent before proceeding with ahi upgrade
  func_logit "Stopping ATB agent and monitor..."
  killall -9 agilent_app_monitor 2>&1 > /dev/null
  killall -9 atb-protocol-agent 2>&1 > /dev/null

}

func_shelfManagerAlive()
{
    : ENTERING: func_shelfManagerAlive
    clia version 2>&1 > /dev/null
    if [ "$?" = "0" ]
    then
      echo "true"
    else
      echo "false"
    fi
}

func_startShelfManager()
{
  : ENTERING: func_startShelfManager
  func_logit "INSTALL: Starting Shelf Manager..."
  ( . /etc/profile && daemon -f shelfman)  # DEN not sourcing profile causes reboot exception
  sleep 5
  func_logit "INSTALL: Shelf Manager Started."

}


func_stopShelfManager()
{
  : ENTERING: func_stopShelfManager
  # Disable crm event system
  rm -rf "$eventDir";
  func_logit "INSTALL: Stopping Shelf Manager...";

  # Before we do ahi programming - which shuts down the Shelf Manager, let's
  # turn up the fans and shut down the blades to prevent them any overheating
  # scenario

  # Turn up fans
#  func_logit -p user.err "INSTALL: $lLabel: $lOption: Adjusting fan speed";

  clia setfanlevel all 5

  # Deactivate necessary blades
#  targetList=`getAmpBlades`

# DEN Need to handle case when a compute blade may be in hub slot 1
#

  case "$gchassisType" in
    F2A|F2AX)
      targetList="2";
      ;;

    F5AX)
      targetList="2 3 4 5";
      ;;

    F14A|F14AX)
      targetList="1 2 3 4 5 6 9 10 11 12 13 14";
      ;;

    *)
      func_logit "INSTALL: ERROR: Failed to match chassis type (ssm)"
      exit 1
      ;;

  esac
  
  if [ ! -z "$targetList" ]
  then
    for blade in $targetList; 
    do 
       func_logit "INSTALL: Deactivating slot $blade"
       clia deactivate board $blade 2>&1 > /dev/null
    done
  fi

  clia terminate
  func_logit "INSTALL: Shelf Manager stopped.";
}



################################################################################
#
# Function: func_Upgrade_USB
# Description: USB FPGA upgrade
#
################################################################################
func_Upgrade_USB()
{

  : ENTERING: func_Upgrade_USB $1 $2
  lAction=$1
  lOption=$2
  lLabel="USB"

  func_stopATB
  
  if [ -z "$chainLength" ]
  then
    func_getXSVF_Filenames
  fi

  if [ "$chainLength" != "4" ]
  then
    return
  fi

  func_check_usb
  # Want this to be silent
  if [ "$gbool_skip_usb" = "true" ] && [ "$gbool_skip_usb_but_erase" = "false" ]
  then
     return
  fi

  func_logit "INSTALL: $lLabel: Action: $lAction $lOption";


  if [ "$lAction" = "check_version" ]
  then

    usbPackage=`cat $pkgVersionFile | grep "usb_prog_4:" | awk '{print $2}'`
    [ -z "$usbPackage" ] && usbPackage="NA"
    usbFirm="unknown"  #DEN no way to determine yet...
     
    if [ "$lOption" = "force" ] || [ "$lOption" = "test" ]
    then
      usbPackage="${usbPackage}_${lOption}"
    fi

    if [ "$usbFirm" != "$usbPackage" ] #### && [ "$usbFirm" != "unknown" ]
    then
      if $gbool_skip_usb
      then
        gbool_Install_usb=false
      else
        logit "INSTALL: Installing USB per progusb file directive"
        gbool_Install_usb=true
      fi

      if $gbool_skip_usb_but_erase
      then
        logit "INSTALL: Erase USB firmware: progusb file directive"
        gbool_Install_usb=true
      fi

      #DEN THESE ARE ALL PLACEHOLDERS
      func_logit "INSTALL: $lLabel: update required $lOption: (target $usbFirm) != (package $usbPackage)";
    else
      func_logit "INSTALL: $lLabel: version up-to-date: $usbFirm, use force to force update (takes 40 minutes)";
    fi
    return
  fi

  if [ "$lAction" = "verify_package" ] || [ "$lAction" = "install" ]  
  then
  
    
    if [ "$chainLength" != "4" ]
    then
      func_logit "$INSTALL: $lLabel: skipping.  USB option not supported on this ESM.";
      return
    fi

    func_binaryCheck "$usb_erase_tar"
    func_binaryCheck "$usb_prog_tar"
    func_checkFreeSpace "$usb_erase_tar" "$usb_prog_tar"
  
    func_logit "INSTALL: $lLabel:   package verified and ready to install.";

  fi

  if [ "$lAction" = "install" ] && $gbool_skip_usb_but_erase
  then
    logit "INSTALL: Erase USB firmware: progusb file directive"
    func_stopShelfManager
    func_downloadFile "$usb_erase_tar"  "$temp/$usb_erase_tar"
    ./ahi_program -e "$temp/$usb_erase_tar";
    
    result=$?
    if test "$result" != 0; then
      func_startShelfManager
      func_logit -p user.err "INSTALL: ERROR: $lLabel erase failed, exit code ${result}";
      exit 1;
    else
      func_logit "  $lLabel erase succeeded.";
    fi
    rm -f "$temp/$usb_erase_tar";

  elif [ "$lAction" = "install" ]
  then
    
    if $gbool_Install_usb || [ "$lOption" = "force" ] ||  [ "$lOption" = "test" ]
    then
      func_logit "INSTALL: $lLabel: $lOption: This step may take up to 40 minutes--please be patient and do not disrupt";

      # Get files over FTP
      func_downloadFile "$usb_erase_tar"  "$temp/$usb_erase_tar"
      func_downloadFile "$usb_prog_tar"    "$temp/$usb_prog_tar"

      if [ "$lOption" = "test" ]
      then
        func_logit "  Test Only..." 
      else
        if [ "$gbool_skip_usb" = "false" ]
        then
          func_stopShelfManager
          func_logit "  Running $lLabel programmer..."
          ./ahi_program -e "$temp/$usb_erase_tar" -p "$temp/$usb_prog_tar";
        else
          logit "INSTALL: Skipping USB install step: progusb file directive"
          true
        fi
      fi

      result=$?
      if test "$result" != 0; then
         func_startShelfManager
         func_logit -p user.err "INSTALL: ERROR: $lLabel firmware upgrade failed, exit code ${result}";
         exit 1;
      else
        func_logit "  $lLabel upgrade succeeded.";
      fi
        
      # Verification of the USB upgrade
      usbPackage=`cat $pkgVersionFile | grep "usb_prog_4:" | awk '{print $2}'`
      [ -z "$usbPackage" ] && usbPackage="NA"
      usbFirm="unknown"  #DEN no way to determine yet...

      if [ "$usbFirm" != "$usbPackage" ] && [ "$usbFirm" != "unknown" ]
      then
         func_logit -p user.err "INSTALL: ERROR: Failed updating $lLabel FPGA. Current version: $usbFirm (expected version: $usbPackage)";
        exit 1;
      fi;

    else
        func_logit "INSTALL: $lLabel: install not necessary.";
    fi # endif gbool_Install_usb

    # Clean up after thyself...
    rm -f "$temp/$usb_erase_tar";
    rm -f "$temp/$ahi_prog_tar";

  fi # endif lAction = install
  
  # Restart in event system is not rebooted ie nothing to update
  func_startShelfManager

}
# end func_Upgrade_USB




################################################################################
#
# Function: func_Upgrade_ATB
# Description: Ahi FPGA upgrade
#
################################################################################
func_Upgrade_ATB()
{

  : ENTERING: func_Upgrade_ATB $1 $2
  lAction=$1
  lOption=$2
  lLabel="ATB"

  func_logit "INSTALL: $lLabel: Action: $lAction $lOption";
  
#DEN  func_stopATB
  func_getXSVF_Filenames

  if [ "$lAction" = "check_version" ]
  then

    ahiPackage=`cat $pkgVersionFile | grep "ahi_prog_${chainLength}" | awk '{print $2}'`
    [ "$(func_ATBAlive)" = "false" ] && func_startATB
    ahiFirm=`${ROOTDIR}/opt/agilent/bin/atb-read-write R 4 N FPGA_VER -x | tail -1 | awk '{print $4}' | cut -c 1-6`;
     
    if [ "$lOption" = "force" ] || [ "$lOption" = "test" ]
    then
      ahiPackage="${ahiPackage}_${lOption}"
    fi

    if [ "$ahiFirm" != "$ahiPackage" ]
    then
      gbool_Install_ahi=true
      func_logit "INSTALL: $lLabel: update required $lOption: (target $ahiFirm) != (package $ahiPackage)";
    else
      func_logit "INSTALL: $lLabel: version up-to-date: $ahiFirm";
    fi
    return
  fi

  if [ "$lAction" = "verify_package" ] || [ "$lAction" = "install" ]  
  then

    func_binaryCheck "$ahi_erase_tar"
    func_binaryCheck "$ahi_prog_tar"
    func_checkFreeSpace "$ahi_erase_tar" "$ahi_prog_tar"
  
    func_logit "INSTALL: $lLabel:   package verified and ready to install.";

  fi


  if [ "$lAction" = "install" ]
  then
    
    if $gbool_Install_ahi || [ "$lOption" = "force" ]
    then
      func_logit "INSTALL: $lLabel: $lOption: ";
      func_logit "INSTALL: $lLabel: Install step will take approximately 20 minutes...";

      # Get files over FTP
      func_downloadFile "$ahi_erase_tar"   "$temp/$ahi_erase_tar"
      func_downloadFile "$ahi_prog_tar"    "$temp/$ahi_prog_tar"

      if [ "$lOption" = "test" ]
      then
        func_logit "  Test Only..." 
      else
        func_stopShelfManager
        func_stopATB
        func_logit "  Running $lLabel programmer..."
        ./ahi_program -e "$temp/$ahi_erase_tar" -p "$temp/$ahi_prog_tar";
      fi

      result=$?
      if test "$result" != 0; then
         func_startShelfManager
         func_logit -p user.err "INSTALL: ERROR: $lLabel firmware update failed, exit code ${result}";
         exit 1;
      else
        func_logit "  $lLabel upgrade succeeded.";
      fi
        
      # Verification of the Ahi upgrade
      ahiPackage=`cat $pkgVersionFile | grep "ahi_prog_${chainLength}" | awk '{print $2}'`
      [ "$(func_ATBAlive)" = "false" ] && func_startATB && sleep 5
      ahiFirm=`${ROOTDIR}/opt/agilent/bin/atb-read-write R 4 N FPGA_VER -x | tail -1 | awk '{print $4}' | cut -c 1-6`;
      
      if [ "$ahiFirm" = "" ]
      then
        func_logit -p user.err "INSTALL:    activate 2.";
        func_stopATB && sleep 1
        ${ROOTDIR}/opt/agilent/bin/esm-fpga-upgrade -eepromtofpga 1
        func_startATB  && sleep 5
        ahiFirm=`${ROOTDIR}/opt/agilent/bin/atb-read-write R 4 N FPGA_VER -x | tail -1 | awk '{print $4}' | cut -c 1-6`;
      fi

      if [ "$ahiFirm" = "" ]
      then
        func_logit -p user.err "INSTALL:    activate 3.";
        func_stopATB && sleep 1
        ${ROOTDIR}/opt/agilent/bin/esm-fpga-upgrade -eepromtofpga 1
        func_startATB && sleep 5
        ahiFirm=`${ROOTDIR}/opt/agilent/bin/atb-read-write R 4 N FPGA_VER -x | tail -1 | awk '{print $4}' | cut -c 1-6`;
      fi

      if test "$ahiFirm" != "$ahiPackage"; then
        func_logit -p user.err "Warning: upgrading $lLabel FPGA. Current version: $ahiFirm (expected version: $ahiPackage)";
#        exit 1;
      else
        func_logit -p user.err "INSTALL: upgrading $lLabel FPGA successful.";
        gbool_reboot=true
      fi;

    else
        func_logit "INSTALL: $lLabel: install not necessary.";
    fi # endif gbool_Install_ahi

    # Clean up after thyself...
    rm -f "$temp/$ahi_erase_tar";
    rm -f "$temp/$ahi_prog_tar";

  fi # endif lAction = install

  # Restart in event system is not rebooted ie nothing to update
  func_startShelfManager

}



################################################################################
#
# Function: func_Upgrade_IDT
# Description: IDT eeprom update
#
################################################################################
func_Upgrade_IDT()
{

  : ENTERING: func_Upgrade_IDT $1 $2
  lAction=$1
  lOption=$2
  lLabel="ESM Connectivity"

  func_logit "INSTALL: $lLabel: Action: $lAction $lOption";
  
#DEN  func_stopATB
  func_getXSVF_Filenames

  if [ "$lAction" = "check_version" ]
  then

    idtupdate=$(./idt_eeprom.sh -ns -install -idtbootcheck -noload -noidtreset 2>&1 | grep "!=")
    if [ -n "$idtupdate" ]
    then
      echo "INSTALL: $lLabel: $idtupdate"
      gbool_Install_idt=true
    fi
  
    idtupdate=$(./idt_eeprom.sh -ns -install -pxiebootcheck -noload -noidtreset 2>&1 | grep "!=")
    if [ -n "$idtupdate" ]
    then
      func_logit "INSTALL: $lLabel: $idtupdate"
      gbool_Install_idt=true
    fi
    if [ "$gbool_Install_idt" != "true" ]
    then
      func_logit "INSTALL: $lLabel: version up-to-date";
    fi
    return
  fi

  if [ "$lAction" = "verify_package" ] || [ "$lAction" = "install" ]  
  then

    for i in eepromw m9502pxi.bin m9505pxi.bin safemode.bin normalmode.bin idt_eeprom.sh
    do
      [ -f "$i" ] || func_logit "INSTALL: $lLabel: File not found: $i"
    done
  fi

  if [ "$lAction" = "install" ]
  then
    
    if $gbool_Install_idt || [ "$lOption" = "force" ]
    then
      func_logit "INSTALL: $lLabel: $lOption: ";
      func_logit "INSTALL: $lLabel: Install step may take up to a few minutes...";

      if [ "$lOption" = "test" ]
      then
        func_logit "  Test Only..." 
      else
        func_stopShelfManager
       
        if [ "$lOption" = "force" ]
        then
          idt_option="-load -noidtreset"
        else
          idt_option="-noidtreset"
        fi

        
        func_logit "INSTALL: $lLabel: Updating IDT EEPROM...";
        ./idt_eeprom.sh -ns -install -idtbootcheck $idt_option
        result=$?
        if test "$result" != 0; then
           func_startShelfManager
           func_logit -p user.err "INSTALL: ERROR: $lLabel idt eeprom configuration update failed, exit code ${result}";
           exit 1;
        fi

        func_logit "INSTALL: $lLabel: Updating PXIe Configuration...";
        ./idt_eeprom.sh -ns -install -pxiebootcheck $idt_option
        result=$?
        if test "$result" != 0; then
           func_startShelfManager
           func_logit -p user.err "INSTALL: ERROR: $lLabel PXIe chassis eeprom update failed, exit code ${result}";
           exit 1;
        fi
        func_logit "INSTALL: $lLabel upgrades succeeded.";
      fi

      # Re Re-Verification of the eeprom upgrade
#      ./idt_eeprom.sh -ns -install -idtbootcheck -noload -verify -noidtreset
#      if test "$result" != 0; then
#         func_startShelfManager
#         func_logit -p user.err "INSTALL: ERROR: $lLabel idt eeprom configuration verification failed, exit code ${result}";
#         exit 1;
#      fi
#      
#      ./idt_eeprom.sh -ns -install -pxiebootcheck -noload -verify -noidtreset
#      if test "$result" != 0; then
#         func_startShelfManager
#         func_logit -p user.err "INSTALL: ERROR: $lLabel PXIe eeprom configuration verification failed, exit code ${result}";
#         exit 1;
#      fi
#
      ./idt_eeprom.sh -ns -webvars >> /tmp/chassis_var
#      gbool_reboot=true

    else
        func_logit "INSTALL: $lLabel: install not necessary.";
    fi # endif gbool_Install_idt

    # Clean up after thyself...
  
  fi # endif lAction = install

  # Restart in event system is not rebooted ie nothing to update
  func_startShelfManager

}


# delete all records of type from file
# goofy problem with mrec_update on checksum error if record does not exist
func_mrec_delete()
{

  atmp1=/tmp/rec1_$$
  atmp2=/tmp/rec2_$$
  rm -f $atmp1 $atmp2

  cp $5 $atmp1

 delete_not_done=true
 for i in 1 2 3 5 6 7
 do
    ./mrec_update $1 $2 $3 $4 $atmp1 $atmp2 2>&1 | grep Error
    if [ "$?" != "0" ]
    then
      echo "deleting records from $5: $2 $3"
      cp $atmp2 $atmp1
    fi
 done
 cp $atmp1 $6
 rm -f $atmp1 $atmp2
}


# patch sequence is frudatar -> mrec_update -> frudataw
func_patch_ekey2()
{
  IPMC_ADDR="$1"   # 0x10
  FRUID="$2"       # 0
  EKEY_VER="$3"    # "A1.0"
  shift; shift; shift

#  AXIE_SRC="$1"    # esmp_axie.bin
#  ATCA_SRC="$2"    # esmp_atca.bin

  # These are old copies during SQ which may not be needed
  FRU_BACKUP=/var/upgrade/${IPMC_ADDR}_${FRUID}_bak.bin
  FRU_BACKUPTXT=/var/upgrade/${IPMC_ADDR}_${FRUID}_bak.txt
  FRU_NEW=/var/upgrade/${IPMC_ADDR}_${FRUID}_new.bin
  FRU_NEWTXT=/var/upgrade/${IPMC_ADDR}_${FRUID}_new.txt
  rm -f $FRU_NEWTXT $FRU_BACKUPTXT
  
  # referenced as of 1.3.21 release
  mkdir -p /var/upgrade/1321
  FRU_BACKUP=/var/upgrade/1321/${IPMC_ADDR}_${FRUID}_bak.bin
  FRU_BACKUPTXT=/var/upgrade/1321/${IPMC_ADDR}_${FRUID}_bak.txt
  FRU_NEW=/var/upgrade/1321/${IPMC_ADDR}_${FRUID}_new.bin
  FRU_NEWTXT=/var/upgrade/1321/${IPMC_ADDR}_${FRUID}_new.txt
  FRU_DIRTY=/var/upgrade/1321/${IPMC_ADDR}_${FRUID}_dirty


  ETMP1=/tmp/fru1_${IPMC_ADDR}.bin
  ETMP2=/tmp/fru2_${IPMC_ADDR}.bin
#  AXIE_FIX=${AXIE_SRC}_new
#  ATCA_FIX=${ATCA_SRC}_new
  rm -f $ETMP1 $ETMP2 $FRU_NEW $FRU_NEWTXT

  if [ "$(func_shelfManagerAlive)" = "false" ]
  then
    func_logit "INSTALL: $lLabel:   ERROR: Failed to start shelf manager"
    gkey_error_cnt=`expr $gkey_error_cnt + 1`
    return
  fi

  # backup originals only once
  [ -f "$FRU_BACKUP" ]    || clia frudatar $IPMC_ADDR $FRUID $FRU_BACKUP
  [ -f "$FRU_BACKUPTXT" ] || clia fruinfo -v $IPMC_ADDR $FRUID > $FRU_BACKUPTXT

  # Revision of Chassis
  shouldUpdatePwr=$(clia fruinfo $IPMC_ADDR $FRUID | grep "Product Version" | egrep -c "ES1|ES2|ES3")
    
  if [ ! -f "$FRU_BACKUP" ]
  then
    func_logit "INSTALL: $lLabel:   ERROR: Failed to backup IPMC: $IPMC_ADDR FRU: $FRUID record, aborting"
    gkey_error_cnt=`expr $gkey_error_cnt + 1`
    return
  fi

  if [ -f "$FRU_DIRTY" ]
  then
    func_logit "INSTALL: $lLabel:   WARNING: IPMC: $IPMC_ADDR FRU: $FRUID recovering from previous update attempt"
    clia frudataw $IPMC_ADDR $FRUID $FRU_BACKUP && rm -f $FRU_DIRTY
  fi

  # inspect backup for goodness
    infoCnt=$(grep -c "Info Area" $FRU_BACKUPTXT)
  
  echo "Deleting records. May get checksum errors when no more records exist"
  if [ "$FRUID" = "0" ]
  then
    if [ "$infoCnt" != "2" ]
    then
      func_logit "INSTALL: $lLabel:   ERROR: ESM FRU section count inconsistent: expected 2, got $infoCnt"
      gkey_error_cnt=`expr $gkey_error_cnt + 1`
      return
    fi

    ## remove old atca board point-to-point record, mfg, rec id, instance
    ./mrec_update -d 0x00315A 0x14 0 $FRU_BACKUP $ETMP1
    ./mrec_update -d 0x00315A 0x14 0 $ETMP1 $ETMP2
    ./mrec_update -d 0x00315A 0x14 0 $ETMP2 $ETMP1

    ## Remove AXIe board ekey records
    func_mrec_delete -d 0x008B19 0x01 0 $ETMP1 $ETMP2
    check_type="Board"
  else
    if [ "$infoCnt" != "3" ]
    then
      func_logit "INSTALL: $lLabel:   ERROR: Backplane FRU section count inconsistent: expected 3, got $infoCnt"
      gkey_error_cnt=`expr $gkey_error_cnt + 1`
      return
    fi
    ## remove old atca backplane connectivity records, mfg, rec id, instance
    ./mrec_update -d 0x00315A 0x04 0 $FRU_BACKUP $ETMP1 
    ./mrec_update -d 0x00315A 0x04 0 $ETMP1 $ETMP2
    ./mrec_update -d 0x00315A 0x04 0 $ETMP2 $ETMP1

    ## Remove AXIe backplane connectivity records
    func_mrec_delete -d 0x008B19 0x00 0 $ETMP1 $ETMP2
    
    # remove power distribution records to make 23amp feed on f5ax
    if [ "$gchassisType" = "F5AX" ] && [ "$shouldUpdatePwr" = "1" ]
    then
       func_logit "INSTALL: $lLabel:   F5AX $check_type power record check";
      ./mrec_update -d 0x00315A 0x11 0 $ETMP2 $ETMP1
      ./mrec_update -d 0x0013BD 0x11 0 $ETMP1 $ETMP2
    fi
    check_type="Backplane"
  fi

  for i in $*
  do
    dd if=$i of=${i}_fix bs=8 skip=1
    ##mrec_update -a <input.bin> <output.bin> <datafile.bin>
    case "$i" in
      F5AX_amp*)
        if [ "$gchassisType" = "F5AX" ] && [ "$shouldUpdatePwr" = "1" ]
        then
          func_logit "INSTALL: $lLabel:   F5AX $check_type updating power record...";
          ./mrec_update -a $ETMP2 $ETMP1 ${i}_fix
          cp $ETMP1 $ETMP2
        fi
        ;;
      *)
        ./mrec_update -a $ETMP2 $ETMP1 ${i}_fix
        cp $ETMP1 $ETMP2

      ;;
    esac
  done

  cp $ETMP2 $FRU_NEW

  touch $FRU_DIRTY
  clia frudataw $IPMC_ADDR $FRUID $FRU_NEW
  clia fruinfo -v $IPMC_ADDR $FRUID > $FRU_NEWTXT 

  # Sanity checks
  # Both contain serial number
  ekey_verify=true
  back_fru=$(grep "Serial Number" $FRU_BACKUPTXT)
  new_fru=$(grep "Serial Number" $FRU_NEWTXT)

  if [ "$back_fru" != "$new_fru" ]
  then
    ekey_verify=false
    func_logit "INSTALL: $lLabel:   ERROR: FRU summary mismatch IPMC: $IPMC_ADDR FRU: $FRUID";
  fi

  new_amps=$(grep "Maximum External Available Current" $FRU_NEWTXT | awk '{print $5}')
  if [ "$checktype" = "Backplane" ] && [ "$gchassisType" = "F5AX" ] && [ "$new_amps" != "23.0" ]
  then
    ekey_verify=false
    func_logit "INSTALL: $lLabel:   ERROR: $check_type power record update failed IPMC: $IPMC_ADDR FRU: $FRUID: expected 23.0 got $new_amps";
  fi
  
  if [ "$checktype" = "Backplane" ] && [ "$gchassisType" = "F5AX" ] && [ "$shouldUpdatePwr" = "1" ]
  then
    new_amps2=$(grep -c "0x0013BD" $FRU_NEWTXT)
    if [ "$new_amps2" -eq 0 ]
    then
      ekey_verify=false
      func_logit "INSTALL: $lLabel:   ERROR: $check_type power record 2 update failed IPMC: $IPMC_ADDR FRU: $FRUID: got $new_amps2";
    fi
  fi


  MRecNew="$(grep -c "$check_type Point-to-Point" $FRU_NEWTXT)"
  if [ "$MRecNew" -ne "2" ] && [ "$MRecNew" -ne "3" ]
  then
    ekey_verify=false
    func_logit "INSTALL: $lLabel:   ERROR: $check_type point-to-point record miscount IPMC: $IPMC_ADDR FRU: $FRUID: got $MRecNew";
  fi

  if $ekey_verify
  then
    patch_fru_info $IPMC_ADDR $FRUID p AssetTag $EKEY_VER
    func_logit "INSTALL: $lLabel:   E-keying update successful for IPMC: $IPMC_ADDR FRU: $FRUID";
    rm -f $FRU_DIRTY

  else
    clia frudataw $IPMC_ADDR $FRUID $FRU_BACKUP
    func_logit "INSTALL: $lLabel:   ERROR: E-keying update failed, aborting change for IPMC: $IPMC_ADDR FRU: $FRUID";
    gkey_error_cnt=`expr $gkey_error_cnt + 1`
    rm -f $FRU_DIRTY
  fi
}


################################################################################
#
# Function: func_Upgrade_Ekey
# Description: Update chassis and ESM for E-key support
#
################################################################################
func_Upgrade_Ekey()
{

  : ENTERING: func_Upgrade_Ekey $1 $2
  lAction=$1
  lOption=$2
  lLabel="E-key Configuration"

  func_logit "INSTALL: $lLabel: Action: $lAction $lOption";
  if [ "$(func_shelfManagerAlive)" = "false" ]
  then
    func_startShelfManager
  fi

  ekeyESMPackage=`cat $pkgVersionFile | grep "esm_ekey:" | awk '{print $2;exit}'`
  ekeyBplPackage=`cat $pkgVersionFile | grep "${gchassisType}_ekey:" | awk '{print $2;exit}'`

  ekeyESMInstalled=$(clia fruinfo 0x10 0  | awk '/Asset Tag/ {print $4}')
  ekeyBplInstalled1=$(clia fruinfo 0x20 1 | awk '/Asset Tag/ {print $4}')
  ekeyBplInstalled2=$(clia fruinfo 0x20 2 | awk '/Asset Tag/ {print $4}')

  if [ "$lAction" = "check_version" ]
  then
    if [ "$ekeyBplPackage" != "$ekeyBplInstalled1" ] 
    then
      func_logit "INSTALL: $lLabel: Chassis ${gchassisType} update required $lOption: (target $ekeyBplInstalled1) != (package $ekeyBplPackage)"
      gbool_Install_ekey=true
    fi
    
    if [ "$ekeyESMPackage" != "$ekeyESMInstalled" ] 
    then
      func_logit "INSTALL: $lLabel: ESM update required $lOption: (target $ekeyESMInstalled) != (package $ekeyESMPackage)"
      gbool_Install_ekey=true
    fi

    if [ "$gbool_Install_ekey" != "true" ]
    then
      func_logit "INSTALL: $lLabel: version up-to-date";
    fi
    return
  fi

  if [ "$lAction" = "verify_package" ] || [ "$lAction" = "install" ]  
  then
    # check for update files
    for i in mrec_update esm_axie.bin esm_atca.bin F5AX_axie1.bin F5AX_axie2.bin F5AX_atca.bin F2AX_axie.bin F2AX_atca.bin F5AX_amp23.bin F5AX_amp23b.bin 
    do
      if [ ! -f "$i" ]
      then
        func_logit "INSTALL: $lLabel: File not found: $i"
        lAction="verify_package"
      fi
    done
  fi

  if [ "$lAction" = "install" ]
  then

    if [ "$gbool_Install_ekey" = "true" ] || [ "$lOption" = "force" ]
    then
      func_logit "INSTALL: $lLabel: $lOption: ";
      func_logit "INSTALL: $lLabel: Install step may take a minute";

      if [ "$lOption" = "test" ]
      then
        func_logit "  Test Only..." 
      else
        func_logit "INSTALL: $lLabel: Updating E-keying Records...";
        if [ "$ekeyBplPackage" != "$ekeyBplInstalled1" ] || [ "$lOption" = "force" ]
        then
           if [ "$gchassisType" = "F5AX" ]
           then
             func_patch_ekey2 0x20 1 "$ekeyBplPackage" F5AX_amp23.bin F5AX_amp23b.bin F5AX_axie1.bin F5AX_axie2.bin F5AX_atca.bin 
           elif [ "$gchassisType" = "F2AX" ]
           then
             func_patch_ekey2 0x20 1 "$ekeyBplPackage" F2AX_axie.bin F2AX_atca.bin 
           fi
        fi
        
        if [ "$ekeyBplPackage" != "$ekeyBplInstalled2" ] || [ "$lOption" = "force" ]
        then
           if [ "$gchassisType" = "F5AX" ]
           then
             func_patch_ekey2 0x20 2 "$ekeyBplPackage" F5AX_amp23.bin F5AX_amp23b.bin F5AX_axie1.bin F5AX_axie2.bin F5AX_atca.bin 
           elif [ "$gchassisType" = "F2AX" ]
           then
             func_patch_ekey2 0x20 2 "$ekeyBplPackage" F2AX_axie.bin F2AX_atca.bin 
           fi
        fi
  

        if [ "$ekeyESMPackage" != "$ekeyESMInstalled" ] || [ "$lOption" = "force" ]
        then
          func_patch_ekey2 0x10 0 "$ekeyESMPackage" esm_axie.bin esm_atca.bin
        fi
  
        if test "$gkey_error_cnt" != 0; then
          func_logit -p user.err "INSTALL: ERROR: $lLabel failed updating e-keying configuration, errors=${gkey_error_cnt}";
        else
          func_logit "INSTALL: $lLabel upgrades succeeded.";
        fi
      fi
        #      gbool_reboot=true
    else
        func_logit "INSTALL: $lLabel: install not necessary.";
    fi # endif gbool_Install_ekey

    # Clean up after thyself...
  
  fi # endif lAction = install

}



################################################################################
#
# Function: func_Upgrade_OS
# Description: U-boot, kernel, and RFS upgrade
#
################################################################################
func_Upgrade_OS()
{
  : ENTERING: func_Upgrade_OS $1 $2
  lAction=$1
  lOption=$2
  lLabel="OS"

  func_logit "INSTALL: $lLabel: Action: $lAction $lOption";
  if [ "$lAction" = "check_version" ]
  then
    crmFirm=`uname -r    | awk -F ${gchassisClass}. '{print $2}' | awk -F - '{print $1}'`
    
    # Check the CRM hw version - we may want to take further action in the
    # future if the version doesn't match
    hw_version=`prg1060 2>/dev/null | grep '^Version:' | awk '{print $2}'`;
    
    crmPackage=`cat $pkgVersionFile | grep '^Release:' | awk '{print $2}'`
    
    if [ "$lOption" = "force" ] || [ "$lOption" = "test" ]
    then
      crmPackage="${crmPackage}_${lOption}"
    fi

    if [ "$crmFirm" != "$crmPackage" ]
    then
      gbool_Install_OS=true
      func_logit "INSTALL: $lLabel: update required $lOption: (target $crmFirm) != (package $crmPackage)";
    else
      func_logit "INSTALL: $lLabel: version up-to-date: $crmFirm";
    fi
    return
  fi

  if [ "$lAction" = "verify_package" ] || [ "$lAction" = "install" ]  
  then

    # Image names for uboot, rfs, and kernel
    kernel=$(func_lookupInstallFile kernel);
    rootfs=$(func_lookupInstallFile rootfs);
    ubootb=$(func_lookupInstallFile u-boot);

    func_binaryCheck "$ubootb"
    func_binaryCheck "$rootfs"
    func_binaryCheck "$kernel"

    func_logit "INSTALL: $lLabel:   package verified and ready to install.";

  fi

  host=`echo "${ftphostdir}" | cut -d/ -f3 | cut -d@ -f2`;
  rootdir=`echo "${ftphostdir}"/ | awk -F ftp://"${host}"/ '{print $2}'`;
  proto="ftp:${host}:/${rootdir}:anonymous:password"

  logfile="${ROOTDIR}/tmp/rupgrade_tool.log.$$";

  if [ "$lAction" = "install" ]
  then
    
    if $gbool_Install_OS || [ "$lOption" = "force" ]
    then
      func_logit "$lLabel: $lOption: rupgrade_tool -s -v -d -f --hook=erase --u=$ubootb --r=$rootfs --k=$kernel --proto=$proto";
      func_preInstallSetup;
      if [ "$lOption" = "test" ]
      then
        func_logit "  Test Only..." 
      else
        func_logit "Running $lLabel upgrade..."
        func_logit "INSTALL: $lLabel: Install step will take approximately 5 minutes...";

#DEN--all in one load and reboot        rupgrade_tool -s -v -d -f --hook=erase --u=$ubootb --r=$rootfs --k=$kernel --proto=$proto 1>$logfile 2>&1 ;
        rupgrade_tool -C -v -d -f --hook=erase --u=$ubootb --r=$rootfs --k=$kernel --proto=$proto 1>$logfile 2>&1 ;
      fi

      result=$?
      if test "$result" != 0; then
         func_logit -p user.err "INSTALL: ERROR: $lLabel firmware update failed, exit code ${result} -- see $logfile";
         exit 1;
      else
        gbool_complete_rupgrade=true;
        func_logit "$lLabel firmware upgrade succeeded, see $logfile for details";
        gbool_reboot=true
      fi
    else
        func_logit "INSTALL: $lLabel: install not necessary.";
    fi # endif gbool_Install_OS
  fi # endif lAction = install

}



#======================================================
# MAIN: 
# 
#======================================================


#
# GLOBALS
#
: ENTERING: $0 $*

gappLogger='logger -t shm-upgrade ';

chassisTypeFile="${ROOTDIR}/etc/chassis_no";
installedVersionFile="${ROOTDIR}/etc/version";
atbPeekPokeUtil="${ROOTDIR}/opt/agilent/bin/atb-read-write";
#fpgaUpgradeUtil="${ROOTDIR}/opt/agilent/bin/amp-crm-fpga-upgrade";
fpgaUpgradeUtil="${ROOTDIR}/opt/agilent/bin/esm-fpga-upgrade";

normalBootCookie="${ROOTDIR}/var/agilent/normal-boot"
upgradeStatusFile="${ROOTDIR}/var/agilent/upgrade-status"
STATUS_NONVOL_FILE="${ROOTDIR}/var/upgrade/upgradelog.txt"


eventDir="${ROOTDIR}/var/agilent/event"
verbose=false
examineInstall=false
forceInstall=false
gbool_Install_OS=false
gbool_complete_rupgrade=false;

gbool_Install_ahi=false
gbool_Install_ekey=false
gbool_Install_idt=false
gbool_Install_usb=false
gbool_Install_chassis=false
gbool_Install_chassisBackup=false

gbool_reboot=false
gbool_EnvSetupDone=false
gkey_error_cnt=0


pkgVersionFile="./version"


#
# Parse Command Line Arguments
#

eval "set -- $AXIE_INITOPTS "'"$@"'
while true
do case "$1" in
	-D ) set -x  # Debug script enabled
        ;;
	-t ) shift || func_usage
        ROOTDIR=$1 # Set a new root directory
        ;;
	-v ) verbose=true
        ;;
	-e ) examineInstall=true
        ;;
   -f ) forceInstall=true
        ;;
	-- ) break;;
	-* ) func_usage Unknown option "$1";;
	*  )
     ftphostdir=$1
     break;;
   esac
   shift
done

[ "X$1" = "X--" ] && shift

[ "$examineInstall" = "false" ] && echo "" > $STATUS_NONVOL_FILE

# Create a temp dir where we will be downloading all our files
temp="$PWD/tmp";
if ! mkdir -m 700 "${temp}" ; then
  func_logit -p user.err "INSTALL: ERROR: Failed to create dir: ${temp}";
  exit 1;
fi;

# Trap for cleaning and exit -- no reboot
trap 'func_cleanupExitTrap "$temp"' INT TERM QUIT EXIT



# Examine System for upgrade
func_preUpgradeCheck


# Install Components
#   options:
#             check_version [test] 
#                 -Compares installed version to that in package.
#                 - if different, set bool_install to true
#                 - [test] a test case to enforce difference
#             verify_package
#                 -Verify that the package components can be downloaded, are accurate, and ready for install
#             install [ test | force ]
#                 -installs the component if version is different
#                 - [test] a test case to enforce difference--does not actuall install anything
#                 - [force] force installation anyway. mutually exclusive with [test]  

#TEST="test"

rm -f ./config
rm -f ./config2
wget -qO  "config" "$ftphostdir/config" 1>/dev/null 2>&1 ;
  
# Check that the files retrieved are actually present
if [ -f "config" ]
then
  # strip out <cr> just in case...
  tr -d '\15' < config > config2
  func_logit "INSTALL: Sourcing custom install config.";
  chmod 755 config2
  . ./config2
else

  if $examineInstall
  then
    func_logit "INSTALL: Inspecting install package and comparing with current revision."
    func_logit "INSTALL: No firmware updates implemented."
    
    func_Upgrade_Chassis check_version $TEST
    func_Upgrade_Chassis verify_package

    func_checkChassisBackup

    func_Upgrade_ATB check_version $TEST
    func_Upgrade_ATB verify_package

    func_Upgrade_USB check_version $TEST
    func_Upgrade_USB verify_package

    func_Upgrade_IDT check_version $TEST
    func_Upgrade_IDT verify_package

    func_Upgrade_Ekey check_version $TEST
    func_Upgrade_Ekey verify_package

    func_Upgrade_OS check_version $TEST
    func_Upgrade_OS verify_package

  elif $forceInstall
  then
    func_logit "INSTALL: Forcing installation of all components to package revision."
    func_logit "INSTALL: Full installation will take approximately 30 minutes."

    func_Upgrade_Chassis check_version $TEST
    func_Upgrade_Chassis verify_package
    func_Upgrade_Chassis install force

    func_checkChassisBackup
    if $gbool_Install_chassisBackup
    then
      func_Upgrade_Chassis verify_package
      sleep 2
      func_Upgrade_Chassis install force
    fi

    func_Upgrade_ATB check_version $TEST
    func_Upgrade_ATB verify_package
    func_Upgrade_ATB install force

    func_Upgrade_USB check_version $TEST
    func_Upgrade_USB verify_package
    func_Upgrade_USB install force

    func_Upgrade_IDT check_version $TEST
    func_Upgrade_IDT verify_package
    func_Upgrade_IDT install force

    func_Upgrade_Ekey check_version $TEST
    func_Upgrade_Ekey verify_package
    func_Upgrade_Ekey install force
    
    func_Upgrade_OS check_version $TEST
    func_Upgrade_OS verify_package
    func_Upgrade_OS install force
    
    gbool_reboot=true


  else # perform the default install
    func_logit "INSTALL: Normal installation. Only components that differ from package will be updated."
    func_logit "INSTALL: Installation time will take between 5 and 30 minutes."
  

    func_Upgrade_Chassis check_version $TEST
    if $gbool_Install_chassis
    then
      func_Upgrade_Chassis verify_package
      func_Upgrade_Chassis install
      gbool_reboot=true
    fi

    func_checkChassisBackup
    if $gbool_Install_chassisBackup
    then
      func_Upgrade_Chassis verify_package
      sleep 2
      func_Upgrade_Chassis install force
    fi

    func_Upgrade_ATB check_version $TEST
    if $gbool_Install_ahi
    then
      func_Upgrade_ATB verify_package
      func_Upgrade_ATB install
      gbool_reboot=true
    fi
    
    func_Upgrade_USB check_version $TEST
    if $gbool_Install_USB
    then
      func_Upgrade_USB install
      gbool_reboot=true
    fi

    func_Upgrade_IDT check_version $TEST
    if $gbool_Install_idt
    then
      func_Upgrade_IDT verify_package
      func_Upgrade_IDT install
    fi
        
    func_Upgrade_Ekey check_version $TEST
    if $gbool_Install_ekey
    then
      func_Upgrade_Ekey verify_package
      func_Upgrade_Ekey install
    fi

    func_Upgrade_OS check_version $TEST
    if $gbool_Install_OS
    then
      func_Upgrade_OS verify_package
      func_Upgrade_OS install
      gbool_reboot=true
    fi

  fi # end else install

  echo "DONE!!"
fi


if $gbool_reboot
then
   func_cleanupRebootTrap
fi

exit 0;
